/* VIDEO CONSOLE ROUTINES */
/* Compile this with Turbo-C command line:  tcc -ml namtok.c */
/* for linux gcc namtok.c -lm -o namtok */
/*

        Z<-CS_CASCADE EXP;AZ;K;N;T;DT;VR;X;Y;WW;SC;SZ2;TXT
        |Plot contours of z=u(x,y,t) where t varies with time
        WW<-0 0,SZ<-#screen
        CNT<-1000 #ifndef "$CS_CNT"
        N<-r$CS_LUT |Lookup table
        |Map screen to unit square
        |Split screen into two
        SZ2<-SZ % 1 2 & Y<-,SZ2[1]/(ymiSZ2[0])%1.0*SZ2[0]
        X<-,SZ[0]/:(iSZ2[1])%1.0*SZ[1]
        T<-0.0 & DT<-0.0001 & K<-0
        AA: ->(K>CNT)/XX
        0 r #do EXP
        Z<-Z+0.001
        TXT<-SZ2 r $CS_LUT["N" #av Z #mod N]
        SC<-SZ2 r "N" #av Z #mod 16
        X<-X+0.5
        0 r #do EXP
        Z<-Z+0.001
        TXT<-TXT,SZ2 r $CS_LUT["N" #av Z #mod N]
        SC<-SC, SZ2 r "N" #av Z #mod 16
        TXT #wput WW
        SC  #wput WW
        |WAIT 10
        K++ & ->AA
        XX: Z<-""

        "$CC.FNS"
        Z<- 100*(X2*X2<-X-T)-(Y*Y)-0.3*T*X*Y
        Z<- 79.3*(X2*X2<-X-0.2*T)-(Y*Y*Y)-0.3*T*X*Y
        Z<- 30.3*(X*X)+Y*Y+250*T
        Z<- 54.3*(X*X)+Y*Y+220*T
        Z<- 88.77*(X*X)+Y*Y+200*T
        Z<- 88.77*(X*X*X)-(Y2*Y2<-Y+200*T)-300*T*X
        Z<- 31.77*(X*X*X)-(Y*Y)-300*T*X
        Z<- 31.77*(20*Y*Y*Y)-(X*X)-120*T*X
        Z<- 13.5*(X-Y-0.5-T)*(4*Y-0.5+T)-2600*T
        Z<- 15.77*((Y+240*T)*(Y-350*T)*Y-750*T)-X*X-160*T
        Z<- 21.6*(1o6.14*X+500*T)+(50.2*2oY*Y+T)+2o6.1416*Y-300*T
        Z<- 4.5*(14.1*X*X-T*20)-(27.6*Y*Y+30*T)+750*T
        Z<- 32.3*(-310*T)+(X2*X2<-X-0.5+40*T)+0.6*Y2*Y2<-Y-0.5
        Z<- 23.323*(-310*T)+(X2*X2<-X-0.5+40*T)+0.6*Y2*Y2<-(30*T)+Y-0.5
        Z<- 11.7*(-45*T)+(X2*X2<-X-0.5)+Y2*Y2<-Y-0.5
        Z<- 76.45*(Y2*Y2<-Y-0.5+60*T)-(X*X*X)-178*T*X<-X-0.5
*/

#include <stdio.h>
#include <time.h>
#include <sys/termios.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <math.h>

/* keyboard stuff */


#define MAX_ESEQ        10      /* input escape sequence */

struct  termios tty_line, tty_screen;
int     tty_fd;
int     sk_sp    = 0;   /* analyse escape sequences */
int     sk_num   = 0;   /* numeric parameter    */
int     sk_eterm = 0;   /* terminator */
int     q_break  = 0;   /* user hit CTRL-C */

#define KB_EOF  0XFFFF
int vget_pushed = KB_EOF;

char   *version_string="namtok 16/2/2001";

char    *intro_text[] = {
"               These programs were written to mark",
"               UNESCO      WORLD MATHEMATICS YEAR 2000",
"",
"Maths is a tool for bridging the gap between wealth and poverty across",
"the globe, the greatest challenge of the 21st century",
"Maurizio Iaccarino,Assistant DG for Science, UNESCO."
"---------------------------------------------------------------------------",
"  Organised Gambling transfers money from the poor to the rich",
"---------------------------------------------------------------------------",
"                       CHAOTIC CASCADES",
"    These dynamic displays are based on discrete sampling of the",
"    equipotential curves k = u(x, y, t) where x,y are space and t is time.",
"    The most interesting are the elliptic curves y*y-x*x*x+a*t*x.",
"                       FRACTAL FOUNTAINS",
"    This set of ASCII animations are taken from the boundary of the",
"    Mandelbrot Set. The Mandelbrot set is defined by the convergence",
"    or otherwise of the iterative scheme z<-z^2+c for a given c.",
"",
"   Press CTRL-Q to Exit, ALT-ENTER for full screen, CTRL-H for Help",
"   (C) Tony Goddard, http://www.d4maths.co.uk/rg.htm",
NULL};

char   *usage_text[] = {
"Usage:",
"   namtok {r=rows} {d=delay} {k=cycles} {x=str} ... {other parameters}",
"   c=cols       Always 80",
"   r=rows       rows of screen",
"   d=delay      timing between each frame",
"   k=cycles     number of frames. 10-32000 (k/2 for mandelbrot points)",
"   v=verbose    If set, then print more messages.",
"   a=advert     advert is a message which will appear on the screen",
"                from time to time. If this message contains spaces",
"                the whole 'a=...' string must be in double quotes",
"",
"   x=str        set of pattern numbers. 0-9, or one of '?aimu'.",
"                0-9    pattern number",
"                ?      random pattern",
"                a      display advert string",
"                i      display info screen",
"                u      display this usage message",
"                M,m    Mandelbrot set animation",
"                       use < . > keys to change colour scheme",
"                       use - = + keys to control interframe delay",
"",
"     See the source code namtok.c for more details.",
"     Details at http://www.d4maths.co.uk/mirage/install.htm",
NULL};

#define S_NOHASSLE      0666    /* read & write */

typedef int     FD;

#define BAD_FD          ((FD)-1)

#define DOUBLE  double

typedef struct { DOUBLE x; DOUBLE y; } COMPLEX;

typedef unsigned char BYTE;     /* unsigned 1 byte data */
typedef int      LONG;     /* delay time on fast computers */

extern  double  cos();
extern  double  sin();
extern  double  fmod();
extern  char    *malloc();
extern  int    atoi();

#define L48_MOD(Z) (Z<0.0) ? 48.0 - fmod( -Z, 48.0): fmod (Z, 48.0);

#define OK   0
#define ERROR   (-1)

#define BREAK   0X03    /* CTRL-C */
#define CTRLB   0X02    /* CTRL-B */
#define CTRLD   0X04    /* CTRL-D */
#define CTRLE   0X05    /* CTRL-E */
#define CTRLF   0X06    /* CTRL-F */
#define CTRLH   0X08    /* CTRL-H */
#define CTRLZ   0X1A    /* CTRL-Z */
#define CTRLQ   0X11    /* CTRL-Q */
#define ESC     0X1B    /* escape key */
#define K_WAIT  1000

/*      Unsolicited keyboard input */
#define BRAKE   break_in()

#define V_NR            25
#define V_NC            80
#define M_PI            3.14159265358979323846
#define M_2_SQRTPI      1.12837916709551257390
#define IRAND(X)        (rand()%(X))
#define LUT_SIZE        4
#define CC_NFNS         11
#define MSG_WAIT        10
#define CC_MAXSIZE      8000
#define FF_MAXD         4000
#define FF_MAXABS       9.0
#define FF_MINABS      -9.0
#define FF_NSCALE       15
#define FF_GRAIN        74.55 /* granularity for mandel_walk */
#define FF_GPLUS        M_2_SQRTPI
#define N_WAIT          250000

BYTE   *vram = NULL;     /* video ram */
BYTE   *vs = NULL;      /* virtual screen */
char    vc_name[0X20];
int     qv = 0;         /* verbose */
LONG    n_wait = N_WAIT;
char   *cc_myvc = "/dev/vcsa";
char   *cc_lut=" ";
char   *cc_fns="0123456789A";
char   *cc_str = "im?u??m*";    /* default for x=str */
char   *cc_font8 = "setfont alt-8x8";
char   *cc_ctrld = "sd4";
char   *cc_advert = NULL;
int     cc_cnt = 600;
int     s_nc = 80;
int     s_maxrows = 50;
int     s_nr = 25;      /* can be changed */
int     cc_size;
int     vc_num = 0;     /* default */

int     ff_si = -1;     /* colour or -1 for all */
DOUBLE *ff_x = NULL;
DOUBLE *ff_y = NULL;
char   *ff_str = NULL;
DOUBLE  ff_aff[] = {0.0, 0.0, 2.0, 2.0};
COMPLEX ff_zc = {0.0, 0.0};
DOUBLE  ff_scale[] =
        {0.25,  0.125, 0.1,   0.1,  0.075,
         0.05,  0.05,  0.025, 0.01,  0.01,
         0.0075,0.005, 0.003, 0.002, 0.001};
DOUBLE  ff_resolution = 74.55;  /* graininess factor for mandel_walk */

main(argc, argv)
int argc;
char **argv;

{
char   *s;
int     rc;

if (NULL ==(vs = malloc(CC_MAXSIZE))) die("memory allocation");
if (NULL ==(vram = malloc(CC_MAXSIZE))) die("memory allocation");

if (NULL ==(ff_x = (DOUBLE*) malloc(FF_MAXD * sizeof (DOUBLE))))
    die("memory allocation X");
if (NULL ==(ff_y = (DOUBLE*) malloc(FF_MAXD * sizeof (DOUBLE))))
    die("memory allocation Y");

tty_init();

if (argc == 1) {
    sprintf(vc_name, "%s%c", cc_myvc, '0'+vc_num);
    printf("virtual console= %s\n", vc_name);
    cc_size = 2*s_nr*s_nc;
    cc_set_rand();
    rc = chaos_cascade(cc_str, cc_cnt);
    cc_repaste(cc_advert, 14);
    cc_msg(intro_text);
    tty_set(0);
    return rc;
    }

for(argv++, argc--; 0 < argc--; argv++) {
    s = *argv;
    if (s[1] == '=') switch(s[0]) {
        case '8':   /* path of font8 .com file */
                  cc_font8 = s+2;
        break;
        case 'A':   /* advert */
        case 'a': cc_advert = s+2;
        break;
        case 'B':   /* background colour for mandel walk */
        case 'b': ff_si = atoi(s+2);
        break;
        case 'P':       /* console 0-9 */
        case 'p': vc_num = atoi(s+2);
        break;
        case 'D':       /* delay */
        case 'd': n_wait = atoi(s+2);
        break;
        case 'I':   /* installation check, CTRL-D exit, etc. */
        case 'i': cc_ctrld = s+2;
        break;
        case 'X':   /* commands */
        case 'x': cc_str = s+2;
        break;
        case 'v':       /* verbose    */
        case 'V': qv = atoi(s+2);
        break;
        case 'k':       /* counter */
        case 'K': cc_cnt = atoi(s+2);
        break;
        case 'c':       /* columns */
        case 'C': s_nc = atoi(s+2);
        break;
        case 'r':       /* rows */
        case 'R': s_nr = atoi(s+2);
        break;
        case 's':       /* si = colour */
        case 'S': ff_si = atoi(s+2);
        break;
        default: break;
        }
    else ff_str = s;
    }

/*      initialise      */

sprintf(vc_name, "%s%c", cc_myvc, '0'+vc_num);
printf("virtual console= %s\n", vc_name);

if (qv) {
    cc_msg(intro_text);
    cc_msg(usage_text);
    }

tty_set(1);
cc_size = 2*s_nr*s_nc;
cc_setfont(s_nr);
cc_set_rand();
rc = chaos_cascade(cc_str, cc_cnt);
cc_repaste(cc_advert, 10);
cc_msg(usage_text);
tty_set(0);
return rc;
}

cc_setfont (int nr)     /* set a font for a number of rows */
{
if (nr>25) {
    system("setfont alt-8x8");
    s_nr = 50;
    }
else {
    system("setfont -16");
    s_nr = 25;
    }
cc_size = 2*s_nr*s_nc;
return OK;
}

chaos_cascade(char *str, int cnt)
{
int     key;
int     imax;
int     cc_shift = 0;
int     i, j, k, kz;
int     fc = 0;
DOUBLE  t  = 0.0;
DOUBLE  dt = 0.0001;
char   *xf;
char   *dst, *src;
if (str == NULL) return ERROR;
for (xf = str; 0 != (fc = *xf++);) {
    key = 0;
    switch (fc) {
    case '^':   /* shift next digit */
            cc_shift = 1;
            break;
    case 'a':   /* advert */
            cc_repaste(cc_advert, 1+IRAND(15));
            key = cc_sleep(MSG_WAIT);
            break;
    case 'u':   /* usage message */
    case 'i':   /* info */
            cc_repaste(" ", (fc == 'u') ? 15 : 1+IRAND(15));
            tcur(0, 0);
            cc_msg((fc == 'i') ? intro_text : usage_text);
            key = cc_sleep(MSG_WAIT);
            break;
    case 'm':   /* mandelwalk */
    case 'M':   key = mandel_view(ff_aff, cnt);
            break;
    case '*':   /* repeat */
            xf = str;
            break;
    case '?':   /* random selection */
            fc = cc_fns[IRAND(CC_NFNS)];
            cc_shift = 0;
    default:    /* display */
            if (cc_shift) {
                fc += ('A'-'0');
                cc_shift = 0;
                }
            t  = 0.0;
            dt = 0.0001;
            for (imax = cnt; 0 < imax--; ) {
                mk_cc_screen(fc, vs, t);
                t += dt;
                /* copy to screen */
                cc_screen_update(vs, ff_si);
                cc_wait(n_wait);
                key = inkey(K_WAIT);
                if (key && key <= ' ') break;
                }       /* imax < cnt */
            break;      /* switch on fc */
            }
    if (key == CTRLF && 0 == system (cc_font8)) {       /* font 8 */
        s_nr = 50;
        cc_size = 2 * s_nr * s_nc;
        }
    else if (key == CTRLE) cc_reset_video();
    else if (key == CTRLH) {
        cc_repaste(" ", 14);
        tcur(0,0);
        cc_msg(usage_text);
        key = cc_sleep(30);
        }
    else if (key == CTRLD) {
        cc_repaste(" ",14);
        system(cc_ctrld);
        }
    else if (key == CTRLZ) {
        cc_repaste(" ", 10);
        system("/bin/bash");   /* DOS prompt */
        }
    else if (key == CTRLQ || key == BREAK || BRAKE) return key;
    }
return  OK;
}

cc_screen_update(BYTE *src, int si)
{
int     j = cc_size;
BYTE   *t = vram;
BYTE   *s = src;
if (si < 0) for (; 0 < j--; *t++ = *s++);
else    for (j = s_nr*s_nc; 0 < j--; s +=2, t += 2) {
            t[0] = *s;
            t[1] = si;
            }
cc_refresh(vram, cc_size, vc_name);
return OK;
}

mk_cc_screen(int fc, char *dst, DOUBLE t)
{
int     i, j, kz;
DOUBLE  x, x2, y, y2, z;
DOUBLE  dy = 1.0 / (DOUBLE) s_nr;
DOUBLE  dx = 1.0 / (DOUBLE) s_nc;

switch (fc) {
    case '0':   /* elliptic curve ghosts */
                /* 76.45*(Y2*Y2<-Y-0.5+60*T)-(X*X*X)-178*T*X<-X-0.5 */
    default:
    for (i = 0, y = -0.5; i< s_nr; i++, y += dy)
        for (j = 0, x = -0.5; j < s_nc; j++, x += dx) {
            y2 = y + t * 60;
            z = 76.45 *((y2*y2) - (x*x*x) + x * t * 178);
            kz = (int) L48_MOD(z);
           *dst++ = cc_lut[kz%4];       /* character */
           *dst++ = kz % 16;            /* attribute */
            }
        break;
    case '1':   /* elliptic curves */
                /* Z<- 88.77*(X*X*X)-(Y2*Y2<-Y+200*T)-300*T*X */
    for (i = 0, y = 0.0; i< s_nr; i++, y += dy)
        for (j = 0, x = 0.0; j < s_nc; j++, x += dx) {
            y2 = y + 200*t;
            z = 88.77*((x*x*x) -(y2*y2)+ 300*t*x);
            kz = (int) L48_MOD(z);
           *dst++ = cc_lut[kz%4];       /* character */
           *dst++ = kz % 16;            /* attribute */
            }
        break;
    case '2':   /* elliptic curve  */
                /* Z<- 31.77*(X*X*X)-(Y*Y)-300*T*X */
    for (i = 0, y = 0.0; i< s_nr; i++, y += dy)
        for (j = 0, x = 0.0; j < s_nc; j++, x += dx) {
            z = 31.77*((x*x*x) -(y*y)+300*t*x);
            kz = (int) L48_MOD(z);
           *dst++ = cc_lut[kz%4];       /* character */
           *dst++ = kz % 16;            /* attribute */
            }
        break;
    case '3':   /* flip elliptic curve  */
                /* Z<- 31.77*(20*Y*Y*Y)-(X*X)-120*T*X */
    for (i = 0, y = 0.0; i< s_nr; i++, y += dy)
        for (j = 0, x = 0.0; j < s_nc; j++, x += dx) {
            z = 31.77*((y*y*y) -(x*x)+120*t*x);
            kz = (int) L48_MOD(z);
           *dst++ = cc_lut[kz%4];       /* character */
           *dst++ = kz % 16;            /* attribute */
            }
        break;
    case '4':   /* elliptic curve with drifting roots */
                /* Z<- 15.77*((Y+240*T)*(Y-350*T)*Y-750*T)-X*X-160*T */
    for (i = 0, y = 0.0; i< s_nr; i++, y += dy)
        for (j = 0, x = 0.0; j < s_nc; j++, x += dx) {
            z = 15.77*((y+240*t)*(y-350*t)*(y-750*t)-x*(x-160*t));
            kz = (int) L48_MOD(z);
           *dst++ = cc_lut[kz%4];       /* character */
           *dst++ = kz % 16;            /* attribute */
            }
        break;
    case '5':   /* drifting ellipses */
                /* Z<- 23.323*(-310*T)+(X2*X2<-X-0.5+40*T)      */
                /*     +0.6*Y2*Y2<-(30*T)+Y-0.5                 */
                /*                                              */
    for (i = 0, y = -0.5; i< s_nr; i++, y += dy)
        for (j = 0, x = -0.5; j < s_nc; j++, x += dx) {
            x2 = x + t * 40;
            y2 = y - t * 30;
            z = 23.323 *((x2*x2) + (0.6*y2*y2) - 310*t);
            kz = (int) L48_MOD(z);
           *dst++ = cc_lut[kz%4];       /* character */
           *dst++ = kz % 16;            /* attribute */
            }
        break;
    case '6':   /* drifting conics */
                /* Z<- 100*(X2*X2<-X-140*T)-(Y*Y)-77.3*T*X*Y */
    for (i = 0, y = 0.0; i< s_nr; i++, y += dy)
        for (j = 0, x = 0.0; j < s_nc; j++, x += dx) {
            x2 = x-30.0*t;
            z = 100.0*((x2*x2) -y*(y- 41.3 * t * x));
            kz = (int) L48_MOD(z);
           *dst++ = cc_lut[kz%4];       /* character */
           *dst++ = kz % 16;            /* attribute */
            }
        break;
    case '7':   /* circles */
                /* Z<- 88.77*(X*X)+Y*Y+200*T */
    for (i = 0, y = 0.0; i< s_nr; i++, y += dy)
        for (j = 0, x = 0.0; j < s_nc; j++, x += dx) {
            z = 88.77*((x*x) + y*(y+200*t));
            kz = (int) L48_MOD(z);
           *dst++ = cc_lut[kz%4];       /* character */
           *dst++ = kz % 16;            /* attribute */
            }
        break;
    case '8':   /* flying conic  */
                /* Z<- 100*(X-Y)*(4*Y+7*T*X)+50*T*X */
    for (i = 0, y = -0.5; i< s_nr; i++, y += dy)
        for (j = 0, x = 0.0; j < s_nc; j++, x += dx) {
            z = 80.0*((x-y)*(4*(y+t)+47*t*x)+30*t*x*y);
            kz = (int) L48_MOD(z);
           *dst++ = cc_lut[kz%4];       /* character */
           *dst++ = kz % 16;            /* attribute */
            }
        break;
    case '9':   /* trig functions  */
        /* Z<- 21.6*(1 o 6.14*X+500*T)+(50.2*2oY*Y+T)+2o6.1416*Y-300*T  */
    for (i = 0, y = 0.0; i< s_nr; i++, y += dy)
        for (j = 0, x = 0.0; j < s_nc; j++, x += dx) {
            z  = sin(6.14*(x+50*t)) + 50.2 * cos(y*(y+t));
            z += cos(6.1416*(y-30*t));
            z  = 11.6*z;
            kz = (int) L48_MOD(z);
           *dst++ = cc_lut[kz%4];       /* character */
           *dst++ = kz % 16;            /* attribute */
            }
        break;
    case 'A':   /* lemniscate style curve */
        /* Z<-15.334*(31.3445337*T)+(2*Y*Y<-Y+T)-X2*1-X2<-X*X */
    for (i = 0, y = 0.0; i< s_nr; i++, y += dy)
        for (j = 0, x = 0.0; j < s_nc; j++, x += dx) {
            x2 = x*x;
            y2 = y+7.6*t;
            z  = 31.3445337*t + 2*y2*y2 - x2*(1-x2);
            z  = 15.334*z;
            kz = (int) L48_MOD(z);
           *dst++ = cc_lut[kz%4];       /* character */
           *dst++ = kz % 16;            /* attribute */
            }
        break;
    case 'B':   /* drifting hyperbola */
        /* Z<-17.333*(X+Y+245*T)*(X-Y) */
    for (i = 0, y = 0.0; i< s_nr; i++, y += dy)
        for (j = 0, x = 0.0; j < s_nc; j++, x += dx) {
            z  = 17.333*(x+y+245*t)*(x-y);
            kz = (int) L48_MOD(z);
           *dst++ = cc_lut[kz%4];       /* character */
           *dst++ = kz % 16;            /* attribute */
            }
        break;
        }
return OK;
}


die(char *str)
{
printf("error: %s\n", str);
return ERROR;
}

cc_msg(char **text)
{
char  **xp = text;
while (NULL != *xp) printf ("%s\n", *xp++);
return OK;
}

cc_wait(unsigned long cnt)        /* short wait */
{
while(cnt > 0) cnt--;
return cnt;
}

cc_set_rand()
{
int     cx = time(NULL);
srand(cx);
return cx;
}

cc_sleep(int ns)      /* wait ns seconds < one minute */
{
int     t, t0, t1, key;
t0 = time(NULL);
t1 = t0+ns;
for(;;){
    t = time(NULL);
    if (t > t1) break;
    key = inkey(1000);
    if (key && key <= ' ') return key;
    }
return OK;
}

cc_reset_video()        /* reset the video */
{
s_nr = 25;
cc_size = 2*s_nr*s_nc;
return OK;
}

cc_repaste(char *msg, int si)   /* set characters, colour in screen */
{
BYTE   *s = msg;
BYTE   *t;
int     j, k, n;
n  = (msg == NULL) ? 0 : strlen(msg);
        for (k = n, j = cc_size, t = vram; 0 < j--; t += 2) {
            if (n == 0) t[0] = ' ';        /* character */
            else {
                t[0] = *s++;
                if (0 == k--) k = n, s = msg;
                }
            if (si) t[1] = si;                  /* attribute */
            }
cc_refresh(vram, cc_size, vc_name);
return OK;
}

int     c_mult(COMPLEX *z, COMPLEX *x, COMPLEX *y)
{
/*  (a+ib)(c+id) = (ac-bd) + i (ad+bc) */
if (x == NULL || y == NULL || z == NULL) return ERROR;
z->x = x->x * y->x - x->y*y->y;
z->y = x->x * y->y + x->y*y->x;
return OK;
}

/*
*  Z<-MS_DELTA T
*  |Point on 1 = modulus 1+ sqrt 1-4*z
*  T<-((?1000)*_PI%500) #ifndef "T"
*  Z<-2 1oT
*  Z<-m 0.25*1 0-Z CMULT Z<-Z-1 0
*/

ms_delta(COMPLEX *zres, DOUBLE zt)
{
COMPLEX zu;
zu.x = cos(zt) - 1.0;
zu.y = sin(zt);
c_mult(zres, &zu, &zu);
zres->y = - 0.25 * zres->y;
zres->x = 0.25 * (1.0 - zres->x);
return OK;
}

ms_clyx(DOUBLE *yd, DOUBLE *xd, int nr, int nc) /* clear vectors */
{
int     n = nr*nc;
while (0<n--) {
   *yd++ = 0.0;
   *xd++ = 0.0;
    }
return OK;
}

mandel_view(DOUBLE *at, int cnt)
{
BYTE   *s, *t;
int     i, iter, j, kz;
int     key = 0;
int     z_size = s_nr * s_nc;
COMPLEX z0;
DOUBLE  theta;
DOUBLE *zp, z;
cc_repaste(" ", 14);
ms_clyx(ff_y, ff_x, s_nr, s_nc);
theta = (M_PI/5000.0)*(DOUBLE) IRAND(10000);
ms_delta(&z0, theta);
at[0] = z0.y;
at[1] = z0.x;
at[2] = at[3] = ff_scale[IRAND(FF_NSCALE)];
for (iter = cnt/2; 0<iter--;) {
    mandel_step(ff_y, ff_x, at, s_nr, s_nc);
    for (j = z_size, zp = ff_y, t = vs; 0 < j--;t += 2) {
         z = *zp++;
         z  = ff_resolution * z;
         kz = (int) L48_MOD(z);
         t[0] = cc_lut[kz%4];                   /* character */
         t[1] = (ff_si > 0) ? ff_si : kz % 16;  /* attribute */
          }
    for (j = cc_size, t = vram, s = vs; 0 < j--; *t++ = *s++);
    cc_refresh(vram, cc_size, vc_name);
    cc_wait(n_wait);
    key = inkey(K_WAIT);
    if      (key == '.')  ff_resolution  = FF_GRAIN;
    else if (key == '>')  ff_resolution *= FF_GPLUS;
    else if (key == '<')  ff_resolution *= 1.0/FF_GPLUS;
    else if (key == 'I')  ff_si = 112;
    else if (key == 'W')  ff_si = 7;
    else if (key == 'G')  ff_si = 10;
    else if (key == 'Y')  ff_si = 14;
    else if (key == 'S')  ff_si = -1;
    else if (key && key <= ' ') break;
    }
return key;
}

mandel_step(DOUBLE *yset, DOUBLE *xset, DOUBLE *at, int nr, int nc)
{
int     i, j, k;
DOUBLE *xs, *ys;
DOUBLE cx, cy, x0, y0, dx, dy, y, zx, zy, xa;

dy = at[2] / (DOUBLE) nr;       /* affine transformation  */
dx = at[3] / (DOUBLE) nc;       /* set center of viewport */
y0 = at[0] + 0.5 * at[2] - dy;
x0 = at[1] - 0.5 * at[3];

xs = xset;
ys = yset;
for (cy = y0, i=0; i<nr; i++, cy -= dy)
    for (cx = x0, j=0; j<nc; j++, cx+= dx) {
        zy = *ys;               /* z<-z^2+c */
        zx = *xs;
        y  = 2 * zx * zy;
        zx = zx*zx - zy*zy;
        xa = zx + cx;
        if      (xa > FF_MAXABS) xa = FF_MAXABS;
        else if (xa < FF_MINABS) xa = FF_MINABS;
       *xs++ = xa;
        xa  = y  + cy;
        if      (xa > FF_MAXABS) xa = FF_MAXABS;
        else if (xa < FF_MINABS) xa = FF_MINABS;
       *ys++ = xa;
        }
return OK;
}

t_wait(int key)        /* modify n_wait */
{
if (key == '=')      n_wait = N_WAIT;
else if (key == '+') n_wait = (3 * n_wait)/2;
else if (key == '-') n_wait = (2 * n_wait)/3;
else return key;
return OK;
}

tcur(X,Y)       /* LOCATE X,Y   Text Cursor<-X,Y */
int     X,Y;
{
return 0;
}


tty_init()
{
void    v_break();
int     rc;
tty_fd = fileno(stdin);
tcgetattr(tty_fd, &tty_line);
tty_screen = tty_line;
tty_screen.c_iflag = BRKINT;
tty_screen.c_oflag = tty_line.c_oflag & ~ 0 /* OPOST */ ;
                                                /* from stty */
tty_screen.c_lflag = tty_line.c_lflag & ~ (ICANON|ECHO|IEXTEN);
                                                /* no echo */
tty_screen.c_cc[VTIME]  = 0;
tty_screen.c_cc[VMIN]   = 1;

/* setbuf(stdout, NULL);        */

signal(SIGINT, v_break);                /* vector to v_break() handler */
return  OK;
}

tty_set(mode)
int     mode;
{
int     rc;
rc = (mode) ? tcsetattr(tty_fd, TCSANOW, &tty_screen):
              tcsetattr(tty_fd, TCSANOW, &tty_line);
return  rc;
}

void v_break()  /* SIGINT handler */
{
q_break = 1;
signal(SIGINT, v_break);        /* reset for LINUX */
}

int     break_in()      /* test global variable */
{
int     rc = q_break;
q_break = 0;            /* turn off */
return rc;
}

int     vgetch()
{
int     done, j, k, n;
char    kq[MAX_ESEQ];

/* deal with pushed key */
if (vget_pushed != KB_EOF) {
    k = (unsigned) vget_pushed;
    vget_pushed = KB_EOF;
    return k;
    }

n = read(tty_fd, &kq[0], MAX_ESEQ);
if (n == 1) {   /* normal character */
    k = (unsigned) kq[0];
    return k;
    }
return ESC;
}

int     inkey(int delay) /* test for key or return 0 after delay */
{
BYTE    xx[MAX_ESEQ];
int     k, cnt;

tty_screen.c_cc[VMIN] = 0;   /* just test for keys */
tcsetattr(tty_fd, TCSAFLUSH, &tty_screen);

for (k = 0, cnt = 1+ delay;
     k == 0 && 0 < cnt--;
     k = (read(tty_fd, &xx[0], MAX_ESEQ)) ? xx[0] : 0);

tty_screen.c_cc[VMIN] = 1;    /* wait for a key */
return k;
}

cc_refresh(src, n, vc)      /* refresh console */
BYTE   *src;    /* object */
int     n;      /* number of bytes to write */
BYTE   *vc;     /* virtual console */
{
extern long lseek();
int     rc = 0;
int     k;
BYTE   *s;
FD      fd;     /* file handle */

if(NULL == vc) return ERROR;
if (BAD_FD ==
        (fd = open(vc, O_RDWR, S_NOHASSLE)))
        return ERROR;

/* lseek(fd, 0L, SEEK_SET);  */
lseek(fd, 4L, SEEK_SET);
k = write(fd, src, n);
return close(fd);
}
