/* Newton Exponential Sums
nes -zone y0 x0 y1 x1 -size ny nx -ofile file ...
Frame buffer version. Allocate whole pixmap
Output: header line, then ny*nx pixels

     This program produces pixmaps where each pixel represents the number
of iterations required to get a result (convergence, divergence) from
the iteration:-

            f(z)-w
     z<-z - ------
            f'(z)

     f(z) is a sum of exponential terms
     w is a point in the complex plain.
     The iteration attempts to invert the function f.

     f(z) = sum { s[i]* exp(a[i]*z | i=1..nterms }

     The coefficients s[i] are either all 1 or 1 -1 1 -1 ...
     The values a[i] are  -ln(1) -ln(2) -ln(3) .. by default.
     if fc > 0 then the coefficients are the remaining non option
     arguements.

     In case of no arguements default to mandelbrot set.
*/

#include <ctype.h>
#include <stdio.h>
#include <math.h>

#define ERROR   -1
#define FCMAX   1
#define MAXARGS 128

double  zw[4]   = {-2,-2,2,2};
int     size[2] = {24,80};
int     fc      = 1;            /* function code */
double  eps     = 1.0e-10;      /* -eps epsilon */
int     bailout = 35;           /* maximum iteration */
int     q_help  = 0;            /* usage message */
int     q_alt   = 0;            /* alternating signs */
int     qt      = 0;            /* verbose, trace .. to stderror */
int     nterms  = 0;            /* number of terms */
int     nxargs  = 0;            /* count command line coefficients */
double *cv      = NULL;         /* coefficient vector */
double *zi      = NULL;
double *zr      = NULL;
int    *sn      = NULL;         /* signs */
double xargs[MAXARGS];          /* supplied coefficients */

char   *ofile   = NULL;
char   *lut     = ".123456789ABCDEF";
int     nlut;

int j_orbit(), n_orbit();
int (*func[])() = {j_orbit, n_orbit};

main(argc, argv)
int argc;
char **argv;
{
extern  double  atof(), log();
extern  char   *malloc();
FILE   *ofp = NULL;
char   *pname = *argv;
char   *s;
char   *rup, *t;
int     i, j, k;
double  x, y, dx, dy;

for(argv++, argc--; 0 < argc--; argv++) {
    s = *argv;
    if (s[0] == '-') switch(s[1]) {
        case 'a':   /* alternating sum */
                q_alt = 1;
        break;
        case 'b':       /* -bailout */
                bailout = atoi(*++argv);
                argc--;
        break;
        case 'e':       /* -eps */
                eps = atof(*++argv);
                argc--;
        break;
        case 'f':       /* -function */
                fc = atoi(*++argv);
                if (fc > FCMAX || fc < 0 ) fc = 0;
                argc--;
        break;
        case 'h':       /* help */
                q_help = 1;
        break;
        case 'l':       /* lut */
                lut = *++argv;
                argc--;
        break;
        case 'n':       /* nterms */
                nterms = atof(*++argv);
                argc--;
        break;
        case 'o':       /* ofile */
                ofile = *++argv;
                argc--;
        break;
        case 's':   /* size */
                size[0] = atoi(*++argv);
                size[1] = atoi(*++argv);
                argc -=2;
        break;
        case 'v':   /* verbose */
                qt = 1;
        break;
        case 'z':   /* zone */
                zw[0] = atof(*++argv);
                zw[1] = atof(*++argv);
                zw[2] = atof(*++argv);
                zw[3] = atof(*++argv);
                argc -= 4;
        break;
        default:        /* could be a number */
                if (isdigit(s[1]) && nxargs < MAXARGS)
                    xargs[nxargs++] = atof(s);
        break;
        }
    else if (isdigit(s[0]) && nxargs < MAXARGS)
         xargs[nxargs++] = atof(s);
    }

nlut = strlen(lut);

if (q_help) return usage(pname);

if (ofile) {
    ofp = fopen(ofile, "w");
    if (ofp == NULL) return ERROR;
    }
else    ofp = stdout;

fprintf(ofp,
    "-fc %d -size %d %d -zone %f %f %f %f -eps %f -bailout %d -lut %s\n",
    fc, size[0], size[1], zw[0], zw[1], zw[2], zw[3], eps, bailout, lut);

/* set up coefficient vector */

if (nterms == 0 && nxargs > 0) {
     nterms = nxargs;
     cv     = &xargs[0];
     }
else {
     cv = (double*) malloc( nterms * sizeof(double));
     if (cv == NULL) return ERROR;
     for (i=0; i<nterms; i++) cv[i] = -log(1.0+i);
     }

/* temporary arrays to calculate sums */

zr = (double*) malloc ((2*nterms)*sizeof(double));
if (zr == NULL) return ERROR;
zi = zr+nterms;

/* alternating sum */

sn = (int*) malloc( nterms * sizeof(int));
if (sn == NULL) return ERROR;
for (i=0, k=1; i<nterms; i++) sn[i]=(q_alt && (i & 1)) ? -1 : 1;

if (qt) {       /* verbose */
    printf("coefficients");
    for (i=0; i<nterms;i++) printf(" %d %f", sn[i], cv[i]);
    printf("\n");
    }

if (NULL == (rup = malloc(size[0]*size[1]))) return ERROR;
dy=(zw[2]-zw[0])/(size[0]-1);
dx=(zw[3]-zw[1])/(size[1]-1);
t = rup;
for (i=0, y=zw[0]; i < size[0]; i++, y +=dy) {
    for (j=0, x=zw[1]; j < size[1]; j++, x += dx) {
        k = (func[fc])(y,x);
       *t++ = (k == 0) ? ' ': lut[k % nlut];
        }
    }

/* print the whole pixmap */
t = rup;

for (i=0; i < size[0]; i++) {
    for (j=0; j < size[1]; j++) { fputc(*t++, ofp); }
    fputc('\n', ofp);
    }
if (ofile) fclose (ofp);
return 0;
}

int     usage(char *pn)
{
printf("%s :: exponential sum fractal generator\n", pn);
printf("the program should be distributed with a script\n");
printf("see the calling script for usage information\n");
return  0;
}

/*      dynamic system z<-z^2+c with global parameters eps, bailout */

int     j_orbit(cy, cx)
double  cy, cx;
{
double  x = 0.0, y = 0.0;
double  x1, y1, x2, y2;
int     j;
for(j = 0; j < bailout; j++) {
    x2= x*x; y2=y*y;
    if (4.1 < x2+y2) return j;
    x1 = cx + x2-y2;
    y1 = cy + 2*x*y;
    x = x1;
    y = y1;
    }
return 0;
}

int     n_orbit(cy, cx)
double  cy, cx;
{
extern double sin(), cos(), exp();
double  x = cx, y = cy;
double  a, b, c, d, delta, dx, dy, e;
int     i, j;
for (j=0; j<bailout; j++) {
     for(i=0; i<nterms; i++) {
         e = exp(x*cv[i]);
         zr[i] = e * cos(y*cv[i]);
         zi[i] = e * sin(y*cv[i]);
         }
     for(a=b=c=d=0.0, i=0; i<nterms; i++) {
         a += sn[i] * zr[i];            /* function */
         b += sn[i] * zi[i];
         c += sn[i] * cv[i] * zr[i];    /* derivative */
         d += sn[i] * cv[i] * zi[i];
         }
    a -= cx;                            /* inverse function */
    b -= cy;
    delta = c*c + d*d;
    if (delta < eps) return j;                  /* divergence */
    dx = (a*c-b*d)/delta;
    dy = (a*d+b*c)/delta;
    if (dx*dx+dy*dy < eps*eps) return j;        /* convergence */
    x -= dx;
    y -= dy;
    }
return 0;
}







